#' Integral Condition Index Scaling
#'
#' Consensuates and scales the category scores of a variable.
#'
#' @param data Excel book with its extension.
#' @param esc Vector with the lower and upper limits of the final scale (def: c(1, 5)).
#' @param ... Additional argument when data is read from Google Sheets. ID: Id of the Google Sheets.
#' @return Consensuated scaled median scores.
#' @author Guillermo Correa (gcorrea@unal.edu.co)
#' @export
#' @references Correa-Londoño, G. (2023). Índice de Condición Integral. https://bookdown.org/gcorrea/ICI/.
#' @seealso [ici] for calculating the ICI, and [ici_pond] for consensuating and
#'   scaling the weights of some variables on a complex construct.
#' @examples
#' # NOTICE: Given that Excel files are not R objects, but external files, they cannot be
#' # loaded into memory when ici package is attached. So, in order to run examples
#' # you must make Excel files available by locating them on your working directory path.
#' # You can find Excel files in the 'extdata/' directory within the installed package
#' # on your local drive. You can also download them through the book:
#' # https://bookdown.org/gcorrea/ICI/ICIresources.rar.
#' #
#' # Example 1: Scores scaled between 1 and 5 (by default)
#' ici_escal("ici_escal.xlsx")
#'
#' # Example 2: Scores scaled between 0 and 7
#' ici_escal("ici_escal.xlsx", esc = c(0, 7))
#'
#' # Example 3: Same scores for all categories
#' ici_escal("ici_escal.xlsx", sheet = 3)
ici_escal <- function (data = NULL, esc = c(1, 5), ...)
{
  # Lectura desde un formulario de Google --------------------------------------
  if (is.null(data)) {
    ID <- c(...)
    exp <- length(googlesheets4::gs4_get(ID)[[6]][[1]])
    for (i in 1:exp) {
      if (i == 1) {
        data <- googlesheets4::read_sheet(ID, 1)
        data2 <- matrix(NA, nrow = nrow(data), ncol = exp - 1)
      }
      else
        data2[, i - 1] <- googlesheets4::read_sheet(ID, i)[[2]]
    }
  data <- data.frame(data, data2)
  }

  # Lectura desde un libro de Excel --------------------------------------------
  else {
    sheet <- c(...)
    if(is.null(sheet))
      sheet <- 1
    data <- data.frame(readxl::read_excel(data, sheet))
  }

  var.name  <- names(data)[[1]]
  data.calc <- data[, -1]

  # Poda de valores por fuera de los límites -----------------------------------
  data.calc[(data.calc < esc[1]) | (data.calc > esc[2])] <- NA

  # Cálculo de calificaciones medianas -----------------------------------------
  calif.esc <- apply(data.calc, 1, stats::median, na.rm = T)
  data <- data.frame(data[[1]], calif.esc)
  names(data) <- c(var.name, "score")

  # Escalamiento de las calificaciones -----------------------------------------

    Lii <- min(data[[2]])  # Calificación mínima
    Lsi <- max(data[[2]])  # Calificación máxima
  for (i in 1:nrow(data))
    data[i, 2] <- (esc[2] - esc[1])/(Lsi - Lii) * (data[i, 2] - Lii) + esc[1]
  print(data, row.names = F)
  if (is.nan(data[1, 2]))
    cat ("\n",
         "Warning! All categories have the same median scores!", "\n",
         "Median scores cannot be scaled!")
  output <- data
}
######################        Fin de la Función       #########################|
