#' Radar Chart for a set of Evaluation Units
#'
#' This function is a wrapper for fmsb::radarchart. Normally, it is automatically
#' called from within ici function (radarchart = TRUE). Even though, if some
#' parameters need to be modified, it can be directly called.
#'
#' Draws a radar chart for the dimensions of some evaluation units.
#'
#' @param data Data frame produced by the ici function
#' @param minmax A two values vector with minimum and maximum of the scale
#' @param dim.char Number of dimensions to be plotted in the radar chart
#' @param calc.ici Logical variable indicatig wheter ici was calculated or not
#' @param plty A vector of line types for data borders plot (def: 1)
#' @param cglty Line type for radar grids (def: 1)
#' @param cglcol Line color for radar grids (def: "gray")
#' @param axislabcol Color of radar axis labeles (def: "gray")
#' @param calcex Font size magnification for radar axis labes (def: 0.7)
#' @param transp Tranparency factor for polygons fill (def: 0.1)
#' @param pcol A vector of color codes for data borders and fill
#' @param transp Tranparency factor for polygons fill (def: 0.1)
#' @param ... Other customizable parameters from fmsb::radchart function
#' @return A radar chart for the (k-1)-hierarchy constructs.
#' @author Guillermo Correa (gcorrea@unal.edu.co)
#' @export
#' @references Correa-Londoño, G. (2023). Índice de Condición Integral. https://bookdown.org/gcorrea/ICI/.
#' @seealso [ici] for calculating the ICI, [fmsb::radarchart] for drawing a
#' radar chart.
#' @examples
#' # NOTICE: Given that Excel files are not R objects, but external files, they cannot be
#' # loaded into memory when ici package is attached. So, in order to run examples
#' # you must make Excel files available by locating them on your working directory path.
#' # You can find Excel files in the 'extdata/' directory within the installed package
#' # on your local drive. You can also download them through the book:
#' # https://bookdown.org/gcorrea/ICI/ICIresources.rar.
#' #
#' # Example 1a: Default radar chart por example 1 of the book
#' # This is what ici function makes by default, when calls ici_radarchart
#' res1a <- ici("Eje1m.xlsx", ue = 2, radar = F)  # Creates an output data frame
#' ici_radarchart(res1a, minmax = c(1, 5), dim.char = 4, calc.ici = T)
#'
#' # Example 1b: Default radar chart por example 1 of the book scaled from 0 to 100
#' # This is what ici function makes by default, when calls ici_radarchart
#' res1b <- ici("Eje1m.xlsx", ue = 2, esc = c(1, 5, 0, 100), radar = F)  # Output data frame
#' ici_radarchart(res1b, minmax = c(0, 100), dim.char = 4, calc.ici = T)
#'
#' # Example 1c: A radar chart for the same data, with some customization
#' ici_radarchart(res1b, minmax = c(0, 100), dim.char = 4, calc.ici = T,
#' cglcol = "pink", axislabcol = "red", pcol = c("blue", "orange"),
#' transp = 0.2,
#' vlabels = c("Dim.\nAmbiental", "Dim.\nTécnica",
#'             "Dim.\nSocial","Dim.\nEconómica"), vlcex = 0.7)
#'
#' # Example 2a:  Default radar chart por example 2 of the book
#' res2 <- ici("Eje2m.xlsx", ue = 2, esc = c(1, 5, 0, 100), nc = 4, radar = F)
#' ici_radarchart(res2, minmax = c(0, 100), dim.char = 3, calc.ici = T)
#'
#' # Example 2b: A radar chart for the same data, with some customization
#' ici_radarchart(res2, minmax = c(0, 100), dim.char = 3, calc.ici = T,
#'               cglty = 3, axislabcol = "blue", calcex = 1.2,
#'               transp = 0, plwd = 3)
ici_radarchart <- function (data, minmax, dim.char, calc.ici,
                            plty = 1, cglty = 1, cglcol = "gray",
                            axislabcol = "gray", calcex = 0.7, pcol = NULL,
                            transp = 0.1, ...)
{

  # Dibuja un radar chart para las dimensiones de varias unidades de evaluación

  # Adaptación del data frame --------------------------------------------------

  n.dim <- dim(data)[2]  # Número de columnas del data frame de entrada
  n.ue  <- dim(data)[1]  # Número de unidades de evaluación

  # Poda de columnas del dataframe ***
  if (calc.ici) {
    data  <- data[, -c(1:(n.dim-dim.char-1))]
    data  <- data[, -dim(data)[2]]
  }
  else
    data  <- data[, -c(1:(n.dim-dim.char))]

  n.dim <- dim.char
  data[n.ue + 1, ] <- minmax[2] # Máximo de las escala
  data[n.ue + 2, ] <- minmax[1] # Mínimo de la escala
  data <- data[c(n.ue + 1, n.ue + 2, 1:n.ue), ]

  # Adaptación de parámetros para el diagrama ----------------------------------

  # Lista de colores (40 colores) ***
  if(is.null(pcol)) {  # Si el usuario no suministra otro colores
    pcol <- c("red", "green", "blue", "yellow", "magenta",  "cyan", "darkorange",
              "bisque", "darkgreen", "purple", "darkseagreen1", "firebrick4",
               "khaki", "pink", "darkolivegreen3", "indianred",  "lightcyan3",
               "plum2", "blue4", "bisque3", "yellow4", "darkgoldenrod3",
               "deepskyblue4", "mediumturquoise", "tomato2", "lightgreen",
               "thistle1", "aliceblue", "gold", "cornflowerblue", "deeppink",
               "darkslategray", "deeppink4", "khaki4", "lightblue1",
               "lightpink", "tan", "yellow3", "springgreen4", "violetred")

  if (n.ue < 40)
    pcol <- pcol[1:n.ue]
  }

  # Ajuste de transparencia de los colores ***
  pfcol <- grDevices::adjustcolor(pcol, alpha.f = transp)

  # Definición del número de polígonos concéntricos (segmentos) ***
  seg <- floor(ceiling(minmax[2])-floor(minmax[1]))
  if (seg > 9) {
    div <- seg/5
    seg <- seg/div
  }

  # Etiquetas para los polígonos concéntricos ***
  if ((minmax[2]-minmax[1]) > 9) {
    caxislabels <- as.vector(minmax[1])
    for (i in 1:seg)
      caxislabels[i+1] <- round(minmax[1]+(minmax[2]-minmax[1])/(seg/i))
  }
  else {
    caxislabels <- seq(floor(minmax[1]), ceiling(minmax[2]))
  }
  caxislabels <- as.character(caxislabels)

  # Construcción del gráfico de radar ------------------------------------------
  fmsb::radarchart(data,
                   axistype = 1,
                   seg = seg,
                   pcol = pcol,
                   plty = plty,
                   pfcol = pfcol,
                   cglty = cglty,
                   cglcol = cglcol,
                   axislabcol = axislabcol,
                   centerzero = T,
                   caxislabels = caxislabels,
                   calcex = calcex,
                   ...)
}
######################        Fin de la Función       #########################|
