#' Integral Condition Index
#'
#' Calculates condition indexes over some evaluation units.
#'
#' @param data Excel book with its extension.
#' @param format type of format: m (marks by default) or c (calificaciones (scores)).
#' @param ue Unidad de evaluación.
#' @param esc Scaling.
#' @param nc Number of constructs.
#' @param console Show results on console (def. TRUE).
#' @param radar Draws a radar chart (def. TRUE).
#' @return Calculate the ICI and the itermediate indexes over one or several evaluation units.
#' @author Guillermo Correa (gcorrea@unal.edu.co)
#' @export
#' @references Correa-Londoño, G. (2023). Índice de Condición Integral. https://bookdown.org/gcorrea/ICI/.
#' @seealso [ici_escal] for consensuating and scaling the category scores of a variable,
#'   and [ici_pond] for consensuating and scaling the weights of some variables on a complex construct.
#' @examples
#' # NOTICE: Given that Excel files are not R objects, but external files, they cannot be
#' # loaded into memory when ici package is attached. So, in order to run examples
#' # you must make Excel files available by locating them on your working directory path.
#' # You can find Excel files in the 'extdata/' directory within the installed package
#' # on your local drive. You can also download them through the book:
#' # https://bookdown.org/gcorrea/ICI/ICIresources.rar.
#' #
#' # Example 1a: Marks format (default), 2 evaluation units
#' ici("Eje1m.xlsx", ue = 2)
#'
#' # Example 1b: Scores format, 2 evaluation units, scaling between 0 and 100
#' ici("Eje1c.xlsx", "c", ue = 2, esc = c(1, 5, 0, 100))
#'
#' # Example 1c: Marks format with missing values
#' ici("Eje1mf.xlsx", ue = 2)
#'
#' #' # Example 1d: Scores format with missing values
#' ici("Eje1cf.xlsx", "c", ue = 2)
#'
#' # Example 2a: 2 evaluation units, 4 constructs, scaling between 0 and 100
#' #             Marks format
#' ici("Eje2m.xlsx", ue = 2, esc = c(1, 5, 0, 100), nc = 4)
#'
#' # Example 2b: 2 evaluation units, 4 constructs, scaling between 0 and 100
#' #             Scores format
#' ici("Eje2c.xlsx", format = "c", ue = 2, esc = c(1, 5, 0, 100), nc = 4)
ici <- function (data, format = c("m", "c"), ue = 1, esc = NULL, nc = 2,
                   console = T, radar = T)
{
  # Calcula el índice de condición integral de una o más unidades de evaluación
  #
  # Args:
  # data    : Nombre del libro Excel con su correspondiente extensión
  # format :   m: marcas (defecto)
  #             c: calificaciones
  # eu      : Unidades de evaluación  (def: 1)
  # esc     : Vector con mínimos y máximos de las dos escalas: mi, Mi, mf, Mf
  #           por ejemplo: esc = c(1, 5, 0, 100)
  # nc      : Número de constructos (def: 2)
  # console : Salida por consola (def: TRUE)
  # radar   : Construyen un radar chart para las dimesiones (def: TRUE)

  format <- match.arg(format)

  # Inicialización del objeto con las salidas ----------------------------------
  out <- vector(mode = "list", length = nc)

  for (u in 1:ue) {

    # Conversión de marcas en calificaciones -----------------------------------
    if (format == "m") {
      if (u == 1) {
        mark <- readxl::read_excel(data)             # Marcas
        cr   <- readxl::read_excel(data, sheet = 2)  # Calif. de referencia
        p1   <- readxl::read_excel(data, sheet = 3)  # Pond. const. jerarquía 1
        vb   <- mark[!is.na(mark[[1]]), 1]           # Var. básicas extraídas
        chg  <- vector("integer", nrow(vb) + 1)      # Vector de cambios
        chg[1] <- 1
        j <- 2
        for (i in 2:nrow(mark)) {
          if (!is.na(mark[i, 1])) {
            chg[j] <- i
            j <- j + 1
          }
        }
        chg[length(chg)] <- nrow(mark) + 1
      }

      # Asignación de calificaciones -------------------------------------------
      mf <- 0  # Marcas faltantes
      for(j in 1:(length(chg)-1)) {
        if (sum(!is.na(mark[chg[j]:(chg[j+1]-1), u+2])) > 1)
          stop ("Variable '", mark[chg[j], 1], "' has more than 1 mark")
        if (sum(!is.na(mark[chg[j]:(chg[j+1]-1), u+2])) == 0)
          mf <- mf + 1   # Actualización de marcas faltantes
        logico <- !is.na(mark[chg[j]:(chg[j+1]-1), u+2])
        if (sum(logico) == 0)
          vb[j, 2] <- NA
        else
          vb[j, 2]  <- cr[chg[j]:(chg[j+1]-1), 2][logico]
      }
      names(vb) <- names(mark)[c(1, u+2)]

      # Integración de variables calificadas y ponderaciones
      j1 <- tibble::tibble(vb, p1[, 2:ncol(p1)])  # Constructos de jerarquía 1
    }

    if (format == "c") {
      if (u == 1) {
        cal  <- readxl::read_excel(data)  # Calif. constructos de jerarquía 1
        p1   <- readxl::read_excel(data, sheet = 2)  # Pond. const. jerarquía 1
      }
    j1 <- tibble::tibble(cal[, 1], cal[, u+1], p1[, 2:ncol(p1)])
    }

    nom.ue <- (names(j1)[2])  # Nombre de la unidad de evaluación
    dim    <- ncol(p1) - 1    # Número de dimensiones
    nv     <- nrow(j1)        # Número de variables
    cf     <- anyNA(j1[[2]])  # Calificaciones faltantes
    if (cf) {
      j1.t <- j1[, -c(1, 2)]
      j1.t[is.na(j1.t)] <- 0
      j1 <- tibble::tibble(j1[, c(1, 2)], j1.t)
    }
    else
      j1[is.na(j1)] <- 0

    # Inicializacion de data frame con los índices -----------------------------
    ind <- tibble::as_tibble(matrix(NA, nrow = dim, ncol = 2),
                             .name_repair = "minimal")
    ind[, 1] <- names(j1)[3:ncol(j1)]

    if (format == "m" & nc > 1) {
      js <- readxl::read_excel(data, sheet = 4)
      names(ind) <- c(names(js)[1], "index")
    }
    else if (format == "c" & nc > 1) {
      js <- readxl::read_excel(data, sheet = 3)
      names(ind) <- c(names(js)[1], "index")
    }
    else {
      nombre <- paste("hierarchy", nc)
      names(ind) <- c(nombre, "index")
    }

    # Cálculo de los indices ---------------------------------------------------
    for (d in 1:dim)
      ind[d, 2] <- stats::weighted.mean(j1[, 2], j1[, d+2], na.rm = T)
    ind[is.nan(ind[[2]]), 2] <- NA

    # Escalamiento de los indices ----------------------------------------------
    if(!is.null(esc))
      for (i in 1:nrow(ind))
        ind[i, 2] <- (esc[4]-esc[3])/(esc[2]-esc[1]) * (ind[i, 2]-esc[1])+esc[3]

    # Salidas Jerarquía 1 ------------------------------------------------------

    # Salida por consola
    if (console) {
      cat("\n", "Condition indexes for", nom.ue, "\n", "\n")
      dig   <- 4
      if (max(ind[, 2], na.rm = T) < 10)
        dig <- 3
      class(ind) <- c("data.frame")
      print(ind, digits = dig, row.names = F)
    }

    # Guardado de salidas en un objeto
    if (u == 1) {
      out[[1]] <- data.frame(t(ind[[2]]))
      names(out[[1]]) <- ind[[1]]
      rownames(out[[1]]) <- nom.ue
    }
    else {
      out[[1]] <- rbind(out[[1]], as.vector(ind[[2]]))
      rownames(out[[1]])[u] <- nom.ue
    }

    dim.char <- dim  # dimensiones para el gráfico de radar
    calc.ici <- F  # No se calcula el ici (para el gráfico de radar)
    # Constructos de jerarquía 2 y siguientes ----------------------------------
    if (nc > 1) {
      for (j in 2:nc) {
        data.c <- js                 # Data frame para cálculos
        dim  <- ncol(data.c) - 1     # Número de dimensiones
        nv   <- nrow(data.c)         # Número de variables
        data.c[is.na(data.c)] <- 0

        if (dim > 1)
          dim.char <- dim  # Actualización de dimensiones para el gráfico de radar
        if (dim == 1)
          calc.ici = T  # Sí se calcula el ici (para el gráfico de radar)

        # Inicialización de data frame con los índices -------------------------
        ind.s <- tibble::as_tibble(matrix(NA, nrow = dim, ncol = 2),
                           .name_repair = "minimal")
        ind.s[, 1] <- names(data.c)[2:ncol(data.c)]
        if (j != nc) {
          if (format == "m")
            js <- suppressMessages(readxl::read_excel(data, sheet = j+3))
          else if (format == "c")
            js <- suppressMessages(readxl::read_excel(data, sheet = j+2))
          nombre <- names(js)[1]
        }
        else
          nombre <- paste("hierarchy", nc)
        names(ind.s) <- c(nombre, "index")

        # Cálculo de los índices jerarquía 2 y siguientes ----------------------
        for (d in 1:dim) {
          ind.s[d, 2] <- stats::weighted.mean(ind[2], data.c[d + 1], na.rm = T)
        }

        # Salidas jerarquía 2 y siguientes -------------------------------------

        # Salidas por consola
        if (console) {
          dig <- 4
          if (max(ind.s[, 2]) < 10)
            dig <- 3
          cat("\n")
          class(ind.s) <- c("data.frame")
          print(ind.s, digits = dig, row.names = F)

          if (!is.null(esc) & j == nc)
            cat("\n", "All indexes have been scaled between",
                esc[3], "and", esc[4], "\n")
        }
        ind <- ind.s

        # Guardado de salidas en un objeto
        if (u == 1) {
          out[[j]] <- data.frame(t(ind.s[[2]]))
          names(out[[j]]) <- ind.s[[1]]
          rownames(out[[j]]) <- nom.ue
        }
        else {
          out[[j]] <- rbind(out[[j]], as.vector(ind.s[[2]]))
          rownames(out[[j]])[u] <- nom.ue
        }
      }
      if (format == "m" && mf > 0)
        cat("\n", "Atention! Missing marks on", mf, "variables!", "\n")
    }
  }

  # Consolidacion de data frame con las salidas --------------------------------
  if (nc > 1) {
    out.df <- out[[1]]
    for (i in 2:nc)
      out.df <- cbind(out.df, out[[i]])
  }
  else
    out.df <- out[[1]]
  out.df <- tibble::tibble(UE = rownames(out.df), out.df)
  output <- out.df

  # Radar chart ----------------------------------------------------------------
  if (radar) {
    if (!is.null(esc))
      minmax <-  esc[3:4]
    else {
      if (format == "m") {
        cr <- cr[, -1]
        minmax <-  c(min(cr, na.rm = T), max(cr, na.rm = T))
      }
      if (format == "c") {
        cal <- cal[, -1]
        minmax <-  c(min(cal, na.rm = T), max(cal, na.rm = T))
      }
    }
    ici_radarchart(output, minmax, dim.char, calc.ici)
  }
  output <- output
}
######################        Fin de la Función       #########################|
